<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreCelebrationPriceItemRequest;
use App\Http\Requests\UpdateCelebrationPriceItemRequest;
use App\Http\Resources\CelebrationPriceItemResource;
use App\Models\CelebrationPriceItem;
use App\Models\LogUpdate;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Lang;

class CelebrationPriceItemController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view("celebration-prices/list");
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreCelebrationPriceItemRequest $request)
    {
        $rawData = $request->only(["checkin", "sku", "bedroom", "bedroomName", "minNights", "cost", "status"]);
        
        $celebrationPriceItem = CelebrationPriceItem::create([
            'checkin' => $rawData["checkin"], 
            'sku' => $rawData["sku"],
            'bedroom' => $rawData["bedroom"], 
            'bedroomName' => $rawData["bedroomName"], 
            'minNights' => $rawData["minNights"], 
            'cost' => $rawData["cost"], 
            'status' => $rawData["status"]
        ]);

        $response = new Response();
        $response->withHeaders(['Content-Type' => "application/json"]);
        $response->setContent($celebrationPriceItem);

        if($celebrationPriceItem->id){
            $response->setStatusCode(200, Lang::get('messages.price_item_store_successful'));
        }else{
            $response->setStatusCode(400, Lang::get('messages.price_item_store_failed'));
        }

        return $response;
    }

    /**
     * Display the specified resource.
     */
    public function show(CelebrationPriceItem $celebrationPriceItem)
    {
        $response = new Response();
        $response->withHeaders(['Content-Type' => "application/json"]);
        $response->setContent($celebrationPriceItem);
        
        if($celebrationPriceItem){
            $response->setStatusCode(200, Lang::get('messages.price_item_show_successful'));
        }else{
            $response->setStatusCode(400, Lang::get('messages.price_item_show_failed'));
        }

        return $response;
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateCelebrationPriceItemRequest $request, CelebrationPriceItem $celebrationPriceItem)
    {
        $rawData = $request->only(["checkin", "bedroomName", "minNights", "cost", "status"]);
        
        $celebrationPriceItemUpdated = $celebrationPriceItem->fill([
            'checkin' => $rawData["checkin"], 
            'bedroomName' => $rawData["bedroomName"], 
            'minNights' => $rawData["minNights"], 
            'cost' => $rawData["cost"], 
            'status' => $rawData["status"]
        ])->save();

        $response = new Response();
        $response->withHeaders(['Content-Type' => "application/json"]);
        $response->setContent($celebrationPriceItem);
        
        if($celebrationPriceItemUpdated){
            $response->setStatusCode(200, Lang::get('messages.price_item_update_successful'));
        }else{
            $response->setStatusCode(400, Lang::get('messages.price_item_update_failed'));
        }

        return $response;
    }

    /**
     * Update Batch the specified resource in storage.
     */
    public function updateBatch(Request $request)
    {
        $items = $request->json()->get('items');
        $userId = $request->json()->get('userId');
        $itemsUpdated = [];

        foreach ($items as $item) {
            $celebrationPriceItem = CelebrationPriceItem::find($item["id"]);

            $celebrationPriceItemUpdated = $celebrationPriceItem->fill([
                'cost' => $item["cost"], 
                'status' => $item["status"]
            ])->save();

            if($celebrationPriceItemUpdated){
               array_push($itemsUpdated, $celebrationPriceItem);
            }
        }
        LogUpdate::create([
            "quantity_items_updated" => count($itemsUpdated),
            "user_id" => $userId,
        ]);

        $response = new Response();
        $response->withHeaders(['Content-Type' => "application/json"]);
        $response->setContent($items);
        
        if(true){
            $response->setStatusCode(200, Lang::get('messages.price_item_update_successful'));
        }else{
            $response->setStatusCode(400, Lang::get('messages.price_item_update_failed'));
        }

        return $response;
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(CelebrationPriceItem $celebrationPriceItem)
    {
        $deleted = $celebrationPriceItem->delete();

        return redirect()
                        ->route("deleteCelebrationPriceItem")
                        ->with("hasMessage", true)
                        ->with("messageType",($deleted === true) ? "success" : "warning")             
                        ->with("message",($deleted === true) ? Lang::get('messages.price_item_delete_successful') : Lang::get('messages.price_item_delete_failed'));
    }

    public function listData(Request $request)
    {    
        $draw = $request->get('draw');
        $start = $request->get('start');
        $length = $request->get('length');

        
        $per_page = $start;
        
        $current_page = $start == 0 ? 1 : (($start/$length))+1 ;
        $starting_point = $start;//($current_page * $per_page) - $per_page;
        $items = CelebrationPriceItem::offset($starting_point)->limit($length)->get();
        $total = $items->count();
        
        $additional = [
            "total" => CelebrationPriceItem::count(),
            "per_page" => $per_page,
            "current_page" => $current_page
        ];
        
        $celebrationPriceItemResource = new CelebrationPriceItemResource($items);

        $celebrationPriceItemResource->additional($additional);

        $response = response()->json($celebrationPriceItemResource);

        return $response;
    }
}