<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Http\Requests\StoreUserRequest;
use App\Http\Requests\UpdateUserRequest;
use App\Models\Permission;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $users = User::all();
        
        $viewBag = ["users" => $users]; 
        return view("users/list", $viewBag);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $permissions = Permission::all();
        
        $viewBag = ["permissions" => $permissions]; 
        return view("users/create", $viewBag);
        
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreUserRequest $request)
    {
        $rawData = $request->only(["name", "email", "password", "user_permission", ""]);
        
        $user = User::create([
            "name" => $rawData["name"],
            "email" => $rawData["email"],
            "password" => Hash::make($rawData["password"]),
            "permission_id" => $rawData["user_permission"]
        ]);

        return redirect()
                    ->route('showUser',$user->id)
                    ->with("hasMessage", true)
                    ->with("messageType",($user->id > 0)? "success" : "warning")             
                    ->with("message",($user->id > 0)? "Usuário criado com sucesso" : "Falha ao criar o usuário");
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        $permissions = Permission::all();
        $hasImage = $this->hasImage($user->id, "photo.jpg") || $this->hasImage($user->id, "photo.png");
        $user->photo  = $this->getImage($user);
        $viewBag = ["permissions" => $permissions, "user" => $user, "hasImage" => $hasImage];
        return view("users.edit", $viewBag);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateUserRequest $request, User $user)
    {
        $rawData = $request->only(["name", "email", "newPassword", "lastPassword", "user_permission"]);
        
        if(isset($rawData["newPassword"]) && $rawData["newPassword"] != "" && Hash::make("newPassword") != $rawData["lastPassword"]){
            $user->fill([
                "name" => $rawData["name"],
                "email" => $rawData["email"],
                "password" => Hash::make($rawData["newPassword"]),
                "permission_id" => $rawData["user_permission"]
            ])->save();
        }  else{
            $user->fill([
                "name" => $rawData["name"],
                "email" => $rawData["email"],
                "permission_id" => $rawData["user_permission"]
            ])->save();
        }

        return redirect()
                    ->route('showUser',$user->id)
                    ->with("hasMessage", true)
                    ->with("messageType",($user->id > 0)? "success" : "warning")             
                    ->with("message",($user->id > 0)? "Usuário editado com sucesso" : "Falha ao editar o usuário");
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        $deleted = $user->delete();

        return redirect()
                        ->route("listUser")
                        ->with("hasMessage", true)
                        ->with("messageType",($deleted === true) ? "success" : "warning")             
                        ->with("message",($deleted === true) ? "Usuário deletado com sucesso" : "Falha ao deletar o usuário.");
    }

    private function uploadImage(){

    }

    private function getImage(User $user){
        $response = "https://placehold.co/150x250";

        if($this->hasImage($user->id, "photo.jpg")){
            $path =  storage_path("app/public/users/".$user->id."/photo.jpg");

            if (!File::exists($path)) {
                return $response;
            }

            $response = asset("storage/users/". $user->id . "/photo.jpg");

        }else if($this->hasImage($user->id, "photo.png")){
            $path =  storage_path("app/public/users/".$user->id."/photo.png");

            if (!File::exists($path)) {
                return $response;
            }

            $response = asset("storage/users/". $user->id . "/photo.png");
        }
        
        return $response;
    }
    
    private function hasImage($itemId, $filename){
        $path =  storage_path("app/public/users/".$itemId."/".$filename);

        if (!File::exists($path)) {
            return false;
        }

        return true;
    }


    public function uploadImages(Request $request, Analysis $analysis)
    {
        $rawData = $request->all();

        $group = mb_strtolower($rawData["group"]);
        $countFiles = $rawData["countFiles"];

        foreach (range(1, intval($countFiles)) as $key => $value) {
            if ($request->hasFile("file-{$value}")) {
                $this->upload($request->file("file-{$value}"), $analysis->id, $group);
            }
        }
    }

    

    private function upload($file, $analysisId, $group)
    {
        $path = "public/analysis/{$analysisId}/{$group}";

        if (!Storage::exists($path)) {
            Storage::makeDirectory($path);
        }
        $time = time();
        $name = "image-question-{$time}";

        // Recupera a extensão do arquivo
        $extension = $file->extension();

        // Define finalmente o nome
        $nameFile = "{$name}.{$extension}";

        $upload = $file->storeAs("public/users/{$analysisId}/{$group}", $nameFile);
    }
}
